<?php
require_once __DIR__ . '/../../includes/db.php';
require_once __DIR__ . '/../../includes/core.php';
if (!isLoggedIn() || getRole() !== 'admin') redirect(BASE_URL . '/login.php');

// Initialize variables
$id = intval($_GET['id'] ?? 0);
$username = $full_name = $email = $role = '';
$assigned_counter_id = '';
$error = '';

// Fetch roles
$resRoles = mysqli_query($conn, "SELECT name FROM roles ORDER BY name");
$roles = mysqli_fetch_all($resRoles, MYSQLI_ASSOC);
// Fallback if roles table empty
if (empty($roles)) {
    $roles = array_map(fn($r) => ['name'=>$r], ['admin','manager','sales','warehouse']);
}
// Fetch counters
$countersRes = mysqli_query($conn, "SELECT id,name FROM counters ORDER BY name");
$counters = mysqli_fetch_all($countersRes, MYSQLI_ASSOC);

// Load existing user for edit
if ($id) {
    $stmt = mysqli_prepare($conn, "SELECT username, full_name, email, role, assigned_counter_id FROM users WHERE id = ?");
    mysqli_stmt_bind_param($stmt, 'i', $id);
    mysqli_stmt_execute($stmt);
    mysqli_stmt_bind_result($stmt, $username, $full_name, $email, $role, $assigned_counter_id);
    mysqli_stmt_fetch($stmt);
    mysqli_stmt_close($stmt);
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $full_name = trim($_POST['full_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $role = $_POST['role'] ?? '';
    $assigned_counter_id = $_POST['assigned_counter_id'] ?? '';
    $password = $_POST['password'] ?? '';

    // Validation
    if ($username === '') {
        $error = 'Username is required.';
    } elseif ($role === '') {
        $error = 'Role is required.';
    } else {
        // Check unique username
        $stmt = mysqli_prepare($conn, "SELECT id FROM users WHERE username = ? AND id <> ?");
        mysqli_stmt_bind_param($stmt, 'si', $username, $id);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_store_result($stmt);
        if (mysqli_stmt_num_rows($stmt) > 0) {
            $error = 'Username already taken.';
        }
        mysqli_stmt_close($stmt);
    }

    if (!$error) {
        $cidVal = ($assigned_counter_id === '' ? null : intval($assigned_counter_id));
        if ($id) {
            // Update existing
            if ($password !== '') {
                $hash = password_hash($password, PASSWORD_DEFAULT);
                $stmt = mysqli_prepare($conn, "UPDATE users SET username=?, full_name=?, email=?, role=?, assigned_counter_id=?, password=? WHERE id=?");
                mysqli_stmt_bind_param($stmt, 'ssssisi', $username, $full_name, $email, $role, $cidVal, $hash, $id);
            } else {
                $stmt = mysqli_prepare($conn, "UPDATE users SET username=?, full_name=?, email=?, role=?, assigned_counter_id=? WHERE id=?");
                mysqli_stmt_bind_param($stmt, 'ssssis', $username, $full_name, $email, $role, $cidVal, $id);
            }
        } else {
            // New user
            if ($password === '') {
                $error = 'Password is required for new users.';
            } else {
                $hash = password_hash($password, PASSWORD_DEFAULT);
                $stmt = mysqli_prepare($conn, "INSERT INTO users (username,password,role,full_name,email,assigned_counter_id) VALUES (?,?,?,?,?,?)");
                mysqli_stmt_bind_param($stmt, 'sssssi', $username, $hash, $role, $full_name, $email, $cidVal);
            }
        }
        if (!$error) {
            mysqli_stmt_execute($stmt) or die('Error saving user: ' . mysqli_error($conn));
            mysqli_stmt_close($stmt);
            redirect(BASE_URL . '/panels/admin/users.php');
        }
    }
}

// Render form
ob_start();
?>
<h1><?= $id ? 'Edit' : 'Add' ?> User</h1>
<?php if ($error): ?>
  <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
<?php endif; ?>
<form method="POST">
  <div class="mb-3">
    <label class="form-label">Username</label>
    <input type="text" name="username" class="form-control" value="<?= htmlspecialchars($username) ?>" required>
  </div>
  <div class="mb-3">
    <label class="form-label">Password<?= $id ? ' (leave blank to keep unchanged)' : '' ?></label>
    <input type="password" name="password" class="form-control" <?= $id ? '' : 'required' ?>>
  </div>
  <div class="mb-3">
    <label class="form-label">Role</label>
    <select name="role" class="form-select" required>
      <option value="">-- Select Role --</option>
      <?php foreach ($roles as $r): ?>
        <option value="<?= htmlspecialchars($r['name']) ?>"<?= $r['name'] === $role ? ' selected' : '' ?>><?= htmlspecialchars(ucfirst($r['name'])) ?></option>
      <?php endforeach; ?>
    </select>
  </div>
  <div class="mb-3">
    <label class="form-label">Full Name</label>
    <input type="text" name="full_name" class="form-control" value="<?= htmlspecialchars($full_name) ?>">
  </div>
  <div class="mb-3">
    <label class="form-label">Email</label>
    <input type="email" name="email" class="form-control" value="<?= htmlspecialchars($email) ?>">
  </div>
  <div class="mb-3">
    <label class="form-label">Assign Counter</label>
    <select name="assigned_counter_id" class="form-select">
      <option value="">-- Unassigned --</option>
      <?php foreach ($counters as $c): ?>
        <option value="<?= $c['id'] ?>"<?= $c['id'] == $assigned_counter_id ? ' selected' : '' ?>><?= htmlspecialchars($c['name']) ?></option>
      <?php endforeach; ?>
    </select>
  </div>
  <button class="btn btn-success"><?= $id ? 'Update' : 'Create' ?> User</button>
  <a href="<?= BASE_URL ?>/panels/admin/users.php" class="btn btn-secondary">Cancel</a>
</form>
<?php
$content = ob_get_clean();
require_once __DIR__ . '/../../includes/layout.php';
?>
